using System;
using System.Reflection;
using UnityEngine;

public static class ComponentExtensions
{
    public static T GetCopyOf<T>(this T comp, T other) where T : Component
    {
        Type type = comp.GetType();
        Type othersType = other.GetType();
		if (othersType.IsSubclassOf(type))
        {
            Debug.LogError($"The type \"{type.AssemblyQualifiedName}\" of \"{comp}\" does not match the type \"{othersType.AssemblyQualifiedName}\" of \"{other}\"!");
            return null;
        }

        BindingFlags flags = BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance | BindingFlags.Default;
        PropertyInfo[] pinfos = type.GetProperties(flags);

        foreach (var pinfo in pinfos)
        {
            if (pinfo.CanWrite)
            {
                try
                {
                    pinfo.SetValue(comp, pinfo.GetValue(other, null), null);
                }
                catch
                {
                    /*
                     * In case of NotImplementedException being thrown.
                     * For some reason specifying that exception didn't seem to catch it,
                     * so I didn't catch anything specific.
                     */
                }
            }
        }

        FieldInfo[] finfos = type.GetFields(flags);

        foreach (var finfo in finfos)
        {
			try
			{
				finfo.SetValue(comp, finfo.GetValue(other));
			}
			catch
			{

			}
        }
        return comp as T;
    }
	public static T GetCopyOfPublic<T>(this T comp, T other) where T : Component
	{
		Type type = comp.GetType();
		Type othersType = other.GetType();
		if (othersType.IsSubclassOf(type))
		{
			Debug.LogError($"The type \"{type.AssemblyQualifiedName}\" of \"{comp}\" does not match the type \"{othersType.AssemblyQualifiedName}\" of \"{other}\"!");
			return null;
		}

		BindingFlags flags = BindingFlags.Public | BindingFlags.Instance;
		PropertyInfo[] pinfos = type.GetProperties(flags);

		foreach (var pinfo in pinfos)
		{
			if (pinfo.CanWrite)
			{
				try
				{
					pinfo.SetValue(comp, pinfo.GetValue(other, null), null);
				}
				catch
				{
					/*
                     * In case of NotImplementedException being thrown.
                     * For some reason specifying that exception didn't seem to catch it,
                     * so I didn't catch anything specific.
                     */
				}
			}
		}

		FieldInfo[] finfos = type.GetFields(flags);

		foreach (var finfo in finfos)
		{
			try
			{
				finfo.SetValue(comp, finfo.GetValue(other));
			}
			catch
			{

			}
		}
		return comp as T;
	}

	public static T AddComponentFrom<T>(this GameObject destination, Component original) where T : Component
	{
		T[] m_List = destination.GetComponents<T>();
		System.Type type = original.GetType();
		System.Reflection.FieldInfo[] fields = type.GetFields();

		foreach (T comp in m_List)
		{
			// If we already have one of them
			if (original.GetType() == comp.GetType())
			{
				foreach (System.Reflection.FieldInfo field in fields)
				{
					field.SetValue(comp, field.GetValue(original));
				}
				return comp;
			}
		}

		// By here, we need to add it
		T copy = destination.AddComponent<T>();

		// Copied fields can be restricted with BindingFlags
		foreach (System.Reflection.FieldInfo field in fields)
		{
			field.SetValue(copy, field.GetValue(original));
		}

		return copy;
	}

	public static Component AddComponentFrom(this GameObject destination, Component original)
	{
		Component[] m_List = destination.GetComponents<Component>();
		System.Type type = original.GetType();
		System.Reflection.FieldInfo[] fields = type.GetFields();

		foreach (Component comp in m_List)
		{
			// If we already have one of them
			if (original.GetType() == comp.GetType())
			{
				foreach (System.Reflection.FieldInfo field in fields)
				{
					field.SetValue(comp, field.GetValue(original));
				}
				return comp;
			}
		}

		// By here, we need to add it
		Component copy = destination.AddComponent(original.GetType());

		// Copied fields can be restricted with BindingFlags
		foreach (System.Reflection.FieldInfo field in fields)
		{
			field.SetValue(copy, field.GetValue(original));
		}

		return copy;
	}

	public static T AddComponent<T>(this GameObject go, T toAdd) where T : Component
    {
        return go.AddComponent<T>().GetCopyOf(toAdd) as T;
    }
}